<?php

namespace MichaelRavedoni\SimpleCommerceSwitzerland\Http\Controllers;

use Illuminate\Routing\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Statamic\View\View;
use Statamic\Facades\Entry;
use Tzsk\Otp\Facades\Otp;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use MichaelRavedoni\SimpleCommerceSwitzerland\Mail\MagicLink;
use DoubleThreeDigital\SimpleCommerce\Facades\Order;
use Statamic\Entries\Entry as EntriesEntry;

class CustomerAuthController extends Controller
{
    /**
     * Register.
     *
     * @return \Illuminate\Http\Response
     */
    public function register(Request $request)
    {
        $data = [
            'title' => 'Register',
            'continue' => $request->session()->get('_previous.url'),
        ];

        if ($request->session()->get('customer_id')) {
            $request->session()->flash('message', __('scs::messages.flash.already_have_account'));
            return redirect(route('account'));
        }

        return (new View)->template('auth/register')->layout('layout')->with($data);
    }

    /**
     * Store.
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $email = $request->email;

        $customerCheckExistance = Entry::query()->where('collection', 'customers')->where('email', $email)->first();
        if ($customerCheckExistance) {
            $request->session()->flash('message', __('scs::messages.flash.email_already_exists'));
            return redirect()->back();
        }

        $customer = Entry::make()->collection('customers')->slug(str($email)->slug('-'))->data([
            'name' => $request->name,
            'email' => $email,
        ]);
        $customer->save();

        return redirect('/account')->with('message', __('scs::messages.flash.registered'));
    }

    /**
     * Login.
     *
     * @return \Illuminate\Http\Response
     */
    public function login(Request $request)
    {
        $data = [
            'title' => 'Login',
            'continue' => $request->session()->get('_previous.url'),
        ];

        if ($request->session()->get('customer_id')) {
            $request->session()->flash('message', __('scs::messages.flash.logged'));
            return redirect(route('account'));
        }

        return (new View)->template('auth/login')->layout('layout')->with($data);
    }

    /**
     * Logout.
     *
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        $request->session()->forget('customer_id');
        return redirect('/')->with('message', __('scs::messages.flash.logged_out'));
    }

    public function account(Request $request)
    {
        $data = [
            'title' => 'Account',
            'customer' => Entry::find($request->session()->get('customer_id')),
            'customerId' => $request->session()->get('customer_id'),
            'orders' => Entry::query()->where('customer', $request->session()->get('customer_id'))->where('is_paid', true)->get(),
        ];

        return (new View)->template('auth/account')->layout('layout')->with($data);
    }

    public function order(Request $request, $order)
    {
        $order = Entry::find($order);
        $data = [
            'title' => $order->title,
            'customer' => Entry::find($request->session()->get('customer_id')),
            'customerId' => $request->session()->get('customer_id'),
            'order' => $order,
        ];

        return (new View)->template('auth/order')->layout('layout')->with($data);
    }

    public function accountUpdate(Request $request)
    {
        $customerId = $request->input('customer');
        $customer = Entry::find($customerId);

        if ($request->input('name')) {$customer->set('name', $request->input('name'));}
        if ($request->input('email')) {$customer->set('email', $request->input('email'));}
        if ($request->input('address')) {$customer->set('address', $request->input('address'));}
        if ($request->input('locality')) {$customer->set('locality', $request->input('locality'));}
        if ($request->input('postal_code')) {$customer->set('postal_code', $request->input('postal_code'));}
        if ($request->input('country')) {$customer->set('country', $request->input('country'));}
        if ($request->input('phone')) {$customer->set('phone', $request->input('phone'));}
        $customer->save();
        //dd($customer);

        $request->session()->flash('message', __('scs::messages.flash.account_updated'));
        return redirect()->back();
    }

    /**
     * Check login.
     *
     * @return \Illuminate\Http\Response
     */
    public function loginCheck(Request $request)
    {
        $email = $request->input('email');
        $continue = $request->input('continue');
        $customer = Entry::query()->where('email', $email)->first();
        if ($customer) {
            return $this->sendMagicLink($customer, $continue);
        }

        $request->session()->flash('message', __('scs::messages.flash.customer_not_found'));
        return redirect()->back();
    }

    /**
     * Send magic link.
     *
     * @return \Illuminate\Http\Response
     */
    public function sendMagicLink($customer, $continue)
    {
        $email = $customer->email;
        $patronId = $customer->id();
        $otp = Otp::generate($email);
        $signature = Hash::make($email);
        $url = route('auth.check_magic_link', ['otp' => $otp, 'email' => $email, 'continue' => $continue, 'signature' => $signature]);
        $emailData = ['url' => $url, 'otp' => $otp];
        Mail::to($email)->send(new MagicLink($emailData));

        return redirect(route('auth.enter_magic_link', ['email' => $email, 'continue' => $continue, 'signature' => $signature]));
    }

    /**
     * Display magic link and otp.
     *
     * @return \Illuminate\Http\Response
     */
    public function enterMagicLink(Request $request)
    {
        $data = [
            'title' => 'Enter',
            'email' => $request->input('email'),
            'continue' => $request->input('continue'),
            'signature' => $request->input('signature'),
        ];

        $checkSignature = Hash::check($request->input('email'), $request->input('signature'));
        if (!$checkSignature) {
            $request->session()->flash('message', __('scs::messages.flash.invalid_signature'));
            return redirect()->route('auth.login');
        }

        return (new View)->template('auth/enter')->layout('layout')->with($data);
    }

    /**
     * Check magic link.
     *
     * @return \Illuminate\Http\Response
     */
    public function checkMagicLink(Request $request)
    {
        $otp = $request->input('otp');
        $email = $request->input('email');
        $continue = $request->input('continue', route('account'));
        $signature = $request->input('signature');

        $customer = Entry::query()->where('email', $email)->first();
        $valid = Otp::match($otp, $email);
        $checkSignature = Hash::check($email, $signature);

        if (!$valid) {
            $request->session()->flash('message', __('scs::messages.flash.invalid_code'));
            return redirect()->back();
        }
        if (!$checkSignature) {
            $request->session()->flash('message', __('scs::messages.flash.invalid_signature'));
            return redirect()->back();
        }

        $request->session()->put('customer_id', $customer->id());

        $request->session()->flash('message', __('scs::messages.flash.logged'));
        return redirect($continue);
    }
}
