<?php

namespace MichaelRavedoni\SimpleCommerceSwitzerland\Http\Controllers;

use Illuminate\Routing\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Statamic\Facades\Entry;
use Statamic\Facades\GlobalSet;
use Sprain\SwissQrBill as QrBill;
use Statamic\View\View;
use DoubleThreeDigital\SimpleCommerce\Facades\Order;
use DoubleThreeDigital\SimpleCommerce\Currency;
use Statamic\Facades\Site;

class InvoiceController extends Controller
{
    /**
     * show.
     *
     * @return \Illuminate\Http\Response
     */
    public function showInvoice(Request $request, $orderId)
    {
        $orderEntry = Entry::find($orderId);
        $order = Order::find($orderId);

        if (empty($order)) {
            return "Order not valid.";
        }

        // Create a new instance of QrBill, containing default headers with fixed values
        $qrBill = QrBill\QrBill::create();

        // Add creditor information
        // Who will receive the payment and to which bank account?
        $qrBill->setCreditor(
            QrBill\DataGroup\Element\CombinedAddress::create(
                config('simple-commerce-switzerland.invoice_debitor.name'),
                config('simple-commerce-switzerland.invoice_debitor.address'),
                config('simple-commerce-switzerland.invoice_debitor.postal_code').' '.config('simple-commerce-switzerland.invoice_debitor.locality'),
                config('simple-commerce-switzerland.invoice_debitor.country_code')?? 'CH'
            )
        );

        $qrBill->setCreditorInformation(
            // This is a special QR-IBAN. Classic IBANs will not be valid here.
            QrBill\DataGroup\Element\CreditorInformation::create(
                config('simple-commerce-switzerland.invoice_debitor_iban.iban')
            )
        );

        // Add debtor information
        // Who has to pay the invoice? This part is optional.
        //
        // Notice how you can use two different styles of addresses: CombinedAddress or StructuredAddress.
        // They are interchangeable for creditor as well as debtor.
        $qrBill->setUltimateDebtor(
            QrBill\DataGroup\Element\CombinedAddress::create(
                $orderEntry->billing_name ?? $orderEntry->shipping_name,
                $orderEntry->billing_address ?? $orderEntry->shipping_address,
                ($orderEntry->billing_postal_code ?? $orderEntry->shipping_postal_code).' '.($orderEntry->billing_city ?? $orderEntry->shipping_city),
                $orderEntry->billing_country ?? $orderEntry->shipping_country
            )
        );

        // Add payment amount information
        // What amount is to be paid?
        $qrBill->setPaymentAmountInformation(
            QrBill\DataGroup\Element\PaymentAmountInformation::create(
                'CHF',
                Currency::toDecimal(data_get($order, 'grandTotal'), Site::current())
            )
        );

        // Add payment reference
        // This is what you will need to identify incoming payments.
        if (config('simple-commerce-switzerland.invoice_debitor_iban.type') == 'qr_iban') {
            $besrId = config('simple-commerce-switzerland.invoice_debitor_iban.besr_id');
            $referenceNumber = QrBill\Reference\QrPaymentReferenceGenerator::generate(
                $besrId,  // You receive this number from your bank (BESR-ID). Unless your bank is PostFinance, in that case use NULL.
                str_replace('#', '', $orderEntry->title) // A number to match the payment with your internal data, e.g. an invoice number
            );
            $qrBill->setPaymentReference(
                QrBill\DataGroup\Element\PaymentReference::create(
                    QrBill\DataGroup\Element\PaymentReference::TYPE_QR,
                    $referenceNumber
            ));
        } else {
            // Explicitly define that no reference number will be used by setting TYPE_NON.
            $qrBill->setPaymentReference(
                QrBill\DataGroup\Element\PaymentReference::create(
                    QrBill\DataGroup\Element\PaymentReference::TYPE_NON
                )
            );
        }

        // Optionally, add some human-readable information about what the bill is for.
        $qrBill->setAdditionalInformation(
            QrBill\DataGroup\Element\AdditionalInformation::create(
                'Commande '.$orderEntry->title
            )
        );

        $outputHtml = new QrBill\PaymentPart\Output\HtmlOutput\HtmlOutput($qrBill, 'en');

        //dd($qrBill->getViolations());

        $qrBillHtml = $outputHtml->setPrintable(false)->getPaymentPart();

        $data = [
            'order_entry'  => $orderEntry,
            'order'  => $order,
            'qrBillHtml' => $qrBillHtml,
        ];

        return (new View)->template('scs::invoice')->with($data);
    }

    /**
     * show.
     *
     * @return \Illuminate\Http\Response
     */
    public function showReceipt(Request $request, $orderId)
    {
        $orderEntry = Entry::find($orderId);
        $order = Order::find($orderId);

        if (empty($order)) {
            return "Order not valid.";
        }

        $data = [
            'order_entry'  => $orderEntry,
            'order'  => $order,
        ];

        return (new View)->template('scs::receipt')->with($data);
    }
}
