<?php

namespace Michaelravedoni\StatamicSiteKit;

use Illuminate\Support\Facades\Route;
use Statamic\Statamic;
use Statamic\Facades\Git;
use Statamic\Facades\CP\Nav;
use Statamic\Facades\Permission;
use Statamic\Events\EntryBlueprintFound;
use Statamic\Events\TermBlueprintFound;
use Statamic\Providers\AddonServiceProvider;
use Michaelravedoni\StatamicSiteKit\Events\StatamicSiteKitContentDefaultsSaved;
use Michaelravedoni\StatamicSiteKit\Events\StatamicSiteKitGlobalsUpdated;
use Michaelravedoni\StatamicSiteKit\Events\Redirects\ManualRedirectCreated;
use Michaelravedoni\StatamicSiteKit\Events\Redirects\ManualRedirectDeleted;
use Michaelravedoni\StatamicSiteKit\Events\Redirects\ManualRedirectSaved;
use Michaelravedoni\StatamicSiteKit\Fieldtypes\StatamicSiteKitMetaTitleFieldtype;
use Michaelravedoni\StatamicSiteKit\Fieldtypes\StatamicSiteKitMetaDescriptionFieldtype;
use Michaelravedoni\StatamicSiteKit\Fieldtypes\StatamicSiteKitGooglePreviewFieldtype;
use Michaelravedoni\StatamicSiteKit\Listeners\AppendEntrySeoFieldsListener;
use Michaelravedoni\StatamicSiteKit\Listeners\AppendTermSeoFieldsListener;
use Michaelravedoni\StatamicSiteKit\Listeners\DefaultsSitemapCacheInvalidationListener;
use Michaelravedoni\StatamicSiteKit\Listeners\Subscribers\SitemapCacheInvalidationSubscriber;
use Michaelravedoni\StatamicSiteKit\Modifiers\ParseLocaleModifier;
use Michaelravedoni\StatamicSiteKit\Tags\StatamicSiteKitTags;
use Michaelravedoni\StatamicSiteKit\Commands\UpdateGit as CommandsUpdateGit;
use Michaelravedoni\StatamicSiteKit\Commands\UpdateGitDry as CommandsUpdateGitDry;
use Michaelravedoni\StatamicSiteKit\Middleware\CheckIfSuperToken;

class ServiceProvider extends AddonServiceProvider
{

    public function bootAddon()
    {
        //
    }

    protected $commands = [
        CommandsUpdateGit::class,
        CommandsUpdateGitDry::class,
    ];

    protected $fieldtypes = [
        StatamicSiteKitMetaTitleFieldtype::class,
        StatamicSiteKitMetaDescriptionFieldtype::class,
        StatamicSiteKitGooglePreviewFieldtype::class,
    ];

    protected $listen = [
        EntryBlueprintFound::class => [
            AppendEntrySeoFieldsListener::class,
        ],
        TermBlueprintFound::class => [
            AppendTermSeoFieldsListener::class,
        ],
        StatamicSiteKitContentDefaultsSaved::class => [
            DefaultsSitemapCacheInvalidationListener::class,
        ],
    ];

    protected $middlewareGroups = [
        'statamic.web' => [
            //
        ],
    ];

    protected $modifiers = [
        ParseLocaleModifier::class,
    ];

    protected $routes = [
        'cp' => __DIR__ . '/../routes/cp.php',
        'web' => __DIR__ . '/../routes/web.php',
    ];

    protected $scripts = [
        __DIR__ . '/../public/js/statamic-site-kit.js',
    ];

    protected $stylesheets = [
        __DIR__ . '/../public/css/statamic-site-kit.css',
        __DIR__.'/../resources/css/cp.css'
    ];

    protected $subscribe = [
        SitemapCacheInvalidationSubscriber::class,
    ];

    protected $tags = [
        StatamicSiteKitTags::class,
    ];

    public function boot()
    {
        parent::boot();

        // Set up views path
        $this->loadViewsFrom(__DIR__ . '/../resources/views', 'statamic-site-kit');

        // Set up translations
        $this->loadTranslationsFrom(__DIR__ . '/../resources/lang', 'statamic-site-kit');

        // Load in custom addon config
        $this->mergeConfigFrom(__DIR__ . '/../config/statamic-site-kit.php', 'statamic-site-kit');
        $this->publishes([
            __DIR__ . '/../config/statamic-site-kit.php' => config_path('statamic-site-kit.php'),
        ], 'config');

        // Set up middlewares
        app('router')->aliasMiddleware('super_token', CheckIfSuperToken::class);

        // Set up permissions
        $this->bootPermissions();

        // Set up navigation
        $this->bootNav();

        // Set up git integration
        $this->bootGitListener();
    }

    /**
     * Add our custom navigation items to the CP nav
     *
     * @return void
     */
    public function bootNav()
    {

        if (config('app.env') == 'production' && config('statamic-site-kit.cp_navigation.enabled')) {

            Nav::extend(function ($nav) {
                $nav->remove('Content', 'Globals');
                $nav->remove('Content', 'Assets');
                $nav->remove('Content', 'Taxonomies');
                $nav->remove('Fields');
                $nav->remove('Tools');
            });

            Nav::extend(function ($nav) {
                if (config('statamic-site-kit.cp_navigation.pages')) {
                    $nav->findOrCreate('Content', 'Collections')
                        ->name(__('statamic-site-kit::main.pages'))
                        ->section('Content')
                        ->url('collections/pages')
                        ->children([])
                        ->icon('<svg xmlns="http://www.w3.org/2000/svg" viewBox="0.25 0.25 13.5 13.5" height="24" width="24" stroke-width="0.5"><g><path d="M13.5,7A6.5,6.5,0,1,1,7,.5" fill="none" stroke="#000000" stroke-linecap="round" stroke-linejoin="round"></path><polyline points="10.5 0.5 5.5 5.5 4.5 9.5 8.5 8.5 13.5 3.5" fill="none" stroke="#000000" stroke-linecap="round" stroke-linejoin="round"></polyline></g></svg>');
                }
                if (config('statamic-site-kit.cp_navigation.menus')) {
                    $nav->findOrCreate('Content', 'Navigation')
                        ->name(__('statamic-site-kit::main.menus'))
                        ->section('Content')
                        ->url('navigation')
                        ->icon('<svg xmlns="http://www.w3.org/2000/svg" viewBox="0.25 0.25 13.5 13.5" height="24" width="24" stroke-width="0.5"><g><line x1="13.5" y1="2" x2="0.5" y2="2" fill="none" stroke="#000000" stroke-linecap="round" stroke-linejoin="round"></line><line x1="13.5" y1="7" x2="0.5" y2="7" fill="none" stroke="#000000" stroke-linecap="round" stroke-linejoin="round"></line><line x1="13.5" y1="12" x2="0.5" y2="12" fill="none" stroke="#000000" stroke-linecap="round" stroke-linejoin="round"></line></g></svg>');
                }

                if (config('statamic-site-kit.cp_navigation.edit')) {
                    foreach (config('statamic-site-kit.cp_navigation.edit') as $navitem) {
                        $nav->create(data_get($navitem, 'name'))
                            ->section('Content')
                            ->url(data_get($navitem, 'url'))
                            ->can(data_get($navitem, 'can') ?? 'access cp')
                            ->icon(data_get($navitem, 'icon') ?? '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" fill="currentColor" class="bi bi-collection" viewBox="0 0 16 16">
                            <path d="M2.5 3.5a.5.5 0 0 1 0-1h11a.5.5 0 0 1 0 1h-11zm2-2a.5.5 0 0 1 0-1h7a.5.5 0 0 1 0 1h-7zM0 13a1.5 1.5 0 0 0 1.5 1.5h13A1.5 1.5 0 0 0 16 13V6a1.5 1.5 0 0 0-1.5-1.5h-13A1.5 1.5 0 0 0 0 6v7zm1.5.5A.5.5 0 0 1 1 13V6a.5.5 0 0 1 .5-.5h13a.5.5 0 0 1 .5.5v7a.5.5 0 0 1-.5.5h-13z"/>
                          </svg>');
                    }
                }

            });
        }

        Nav::extend(function ($nav) {

            if (config('statamic-site-kit.cp_navigation.tools')) {
                // Top level Settings item
                $nav->create(__('statamic-site-kit::main.settings'))
                    ->can('configure ssk settings')
                    ->section(config('statamic-site-kit.cp_navigation.tools_section_name') ?? __('statamic-site-kit::main.tools'))
                    ->route('statamic-site-kit.settings.home')
                    ->icon('<svg xmlns="http://www.w3.org/2000/svg" viewBox="0.25 0.25 13.5 13.5" height="600" width="600" stroke-width="0.5"><path d="M5.23,2.25l.43-1.11A1,1,0,0,1,6.59.5h.82a1,1,0,0,1,.93.64l.43,1.11,1.46.84,1.18-.18a1,1,0,0,1,1,.49l.4.7a1,1,0,0,1-.08,1.13L12,6.16V7.84l.75.93a1,1,0,0,1,.08,1.13l-.4.7a1,1,0,0,1-1,.49l-1.18-.18-1.46.84-.43,1.11a1,1,0,0,1-.93.64H6.59a1,1,0,0,1-.93-.64l-.43-1.11-1.46-.84-1.18.18a1,1,0,0,1-1-.49l-.4-.7a1,1,0,0,1,.08-1.13L2,7.84V6.16l-.75-.93A1,1,0,0,1,1.17,4.1l.4-.7a1,1,0,0,1,1-.49l1.18.18ZM5,7A2,2,0,1,0,7,5,2,2,0,0,0,5,7Z" fill="none" stroke="#000000" stroke-linecap="round" stroke-linejoin="round"></path></svg>')
                    ->children([
                        // Informations
                        $nav->item(__('statamic-site-kit::informations.plural'))
                            ->route('statamic-site-kit.settings.informations.index')
                            ->can('view ssk settings informations'),
                        // Manifests
                        $nav->item(__('statamic-site-kit::manifests.plural'))
                            ->route('statamic-site-kit.settings.manifests.index')
                            ->can('view ssk settings manifests'),
                ]);

                // Top level SEO item
                $nav->create(__('statamic-site-kit::main.seo'))
                    ->can('configure ssk seo')
                    ->section(config('statamic-site-kit.cp_navigation.tools_section_name') ?? __('statamic-site-kit::main.tools'))
                    ->route('statamic-site-kit.seo.home')
                    ->icon('seo-search-graph')
                    ->children([
                        // SEO categories
                        $nav->item(__('statamic-site-kit::general.index'))
                            ->route('statamic-site-kit.seo.general.index')
                            ->can('view ssk seo general'),
                        $nav->item(__('statamic-site-kit::defaults.index'))
                            ->route('statamic-site-kit.seo.defaults.index')
                            ->can('view ssk seo defaults'),
                        $nav->item(__('statamic-site-kit::marketing.singular'))
                            ->route('statamic-site-kit.seo.marketing.index')
                            ->can('view ssk seo marketing'),
                        $nav->item(__('statamic-site-kit::social.singular'))
                            ->route('statamic-site-kit.seo.social.index')
                            ->can('view ssk seo social'),
                        $nav->item(__('statamic-site-kit::sitemap.singular'))
                            ->route('statamic-site-kit.seo.sitemap.index')
                            ->can('view ssk seo sitemap'),
                    ]);
            }
        });
    }

    /**
     * Add permissions for Statamic Site Kit
     *
     * @return void
     */
    public function bootPermissions()
    {
        $seo_groups = [
            [
                'value' => 'general',
                'label' => 'General',
            ],
            [
                'value' => 'marketing',
                'label' => 'Marketing',
            ],
            [
                'value' => 'social',
                'label' => 'Social',
            ],
            [
                'value' => 'sitemap',
                'label' => 'Sitemap',
            ],
            [
                'value' => 'defaults',
                'label' => 'Defaults',
            ],
        ];

        Permission::group('statamic-site-kit', 'Statamic Site Kit', function () use ($seo_groups) {
            Permission::register('configure ssk settings', function ($permission) {
                $permission->children([
                    Permission::make('view ssk settings')
                        ->label(__('statamic-site-kit::main.permissions.settings.view'))
                        ->children([
                            Permission::make('view ssk settings informations')
                                ->label(__('statamic-site-kit::informations.permissions.view')),
                            Permission::make('update ssk settings informations')
                                ->label(__('statamic-site-kit::informations.permissions.update')),
                            Permission::make('view ssk settings manifests')
                                ->label(__('statamic-site-kit::manifests.permissions.view')),
                            Permission::make('update ssk settings manifests')
                                ->label(__('statamic-site-kit::manifests.permissions.update')),
                        ]),
                ]);
            })->label(__('statamic-site-kit::main.permissions.settings.configure'));

            Permission::register('configure ssk seo', function ($permission) use ($seo_groups) {
                $permission->children([
                    Permission::make('view ssk seo {seo_group}')
                        ->replacements('seo_group', function () use ($seo_groups) {
                            return collect($seo_groups)->map(function ($group) {
                                return [
                                    'value' => $group['value'],
                                    'label' => $group['label'],
                                ];
                            });
                        })
                        ->label('View :seo_group Settings')
                        ->children([
                            Permission::make('update ssk seo {seo_group}')
                                ->label('Update :seo_group Settings'),
                        ]),
                ]);
            })->label(__('statamic-site-kit::main.permissions.seo.configure'));
        });
    }

    /**
     * Register our custom events with the Statamic git integration
     *
     * @return void
     */
    protected function bootGitListener(): void
    {
        if (config('statamic.git.enabled')) {
            $events = [
                StatamicSiteKitContentDefaultsSaved::class,
                StatamicSiteKitGlobalsUpdated::class,
                ManualRedirectCreated::class,
                ManualRedirectDeleted::class,
                ManualRedirectSaved::class,
            ];

            foreach ($events as $event) {
                Git::listen($event);
            }
        }
    }

    /**
     * Register our custom Schedule
     *
     * @return void
     */
    protected function schedule($schedule)
    {
        $schedule->command(CommandsUpdateGit::class)->hourly();
    }
}
