<?php

namespace Michaelravedoni\StatamicSiteKit\Commands;

use Gitonomy\Git\Repository;
use Illuminate\Console\Command;

class UpdateGitDry extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'git:update-dry';

    /**
     * The console command description.
     */
    protected $description = 'Automatically updates adds, commits and pushes any changes';

    /**
     * Staged files
     */
    public $stagedFiles = [];

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $repository = new Repository('.git');
        $paths = config('statamic.git.paths');

        $this->newLine();
        $this->info('-- Path tracked from `statamic.git.paths` --');
        foreach ($paths as $path) {
            $this->line($this->trimDirectoryName($path));
        }

        $this->newLine();
        $this->info('-- `git status` for `statamic.git.paths` --');
        foreach ($paths as $path) {
            $output = $repository->run('status', ['--short', $path]);
            if ($output) {
                $this->line($output);
                $this->stagedFiles[] = $this->trimFileName($output);
            }
        }

        $this->stagedFiles = array_flatten($this->stagedFiles);

        if (! empty($this->stagedFiles)) {

            $this->newLine();
            $this->info('----------------------------------------------');
            $this->info('| You can run `git:update` to apply changes. |');
            $this->info('----------------------------------------------');
            $this->newLine();
        } else {
            $this->newLine();
            $this->info('--------------');
            $this->info('| No changes |');
            $this->info('--------------');
            $this->newLine();
        }

        $output = $repository->run('status', ['--short']);
        if ($output) {
            $this->newLine();
            $this->info('-- `git status` for all repository --');
            $this->line($output);
        }

        $this->info('-- `git status` for `statamic.git.paths` with ignored and untracked files --');
        foreach ($paths as $path) {
            $output = $repository->run('status', ['--short', '--ignored', $path, '--untracked-files', 'normal']);
            if ($output) {
                $this->line($output);
                //$this->stagedFiles[] = $this->trimFileName($output);
            }
        }

        $this->newLine();
        $this->info('-- Meanings --');
        $this->line('M = modified');
        $this->line('T = file type changed');
        $this->line('! = ignored');
        $this->line('? = untracked');

        return 0;
    }

    public function trimFileName(string $output): array
    {
        $files = [];

        $lines = explode("\n", $output);

        foreach ($lines as $line) {
            $line = trim($line);

            if (strlen($line)) {
                $files[] = basename(substr($line, 2));
            }
        }

        return $files;
    }

    public function trimDirectoryName(string $directory): string
    {
        return str_replace(base_path(), '', $directory);
    }
}
