<?php

namespace Michaelravedoni\StatamicSiteKit\Http\Controllers\CP\Seo;

use Illuminate\Http\Request;
use Statamic\CP\Breadcrumbs;
use Statamic\Facades\Collection;
use Statamic\Facades\Site;
use Statamic\Facades\Taxonomy;
use Michaelravedoni\StatamicSiteKit\Http\Controllers\CP\Controller;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Seo\DefaultsBlueprint;
use Michaelravedoni\StatamicSiteKit\Facades\StatamicSiteKitStorage;
use Michaelravedoni\StatamicSiteKit\Content\ContentDefaults;
use Michaelravedoni\StatamicSiteKit\Events\StatamicSiteKitContentDefaultsSaved;

class DefaultsController extends Controller
{
    /**
     * Display a list of all collections/taxonomies
     */
    public function index()
    {
        $this->authorize('view ssk seo defaults');

        $collections = Collection::all();
        $taxonomies = Taxonomy::all();
        $curr_site = Site::selected();

        $content_types = [
            'Collections' => $collections
                ->filter(function ($collection) use ($curr_site) {
                    return $collection->sites()->contains($curr_site);
                })
                ->map(function ($collection) {
                    return [
                    'count' => $collection->queryEntries()->count(),
                    'handle' => $collection->handle(),
                    'title' => $collection->title(),
                ];
                })->toArray(),
            'Taxonomies' => $taxonomies
                ->filter(function ($taxonomy) use ($curr_site) {
                    return $taxonomy->sites()->contains($curr_site);
                })
                ->map(function ($taxonomy) {
                    return [
                    'count' => $taxonomy->queryTerms()->count(),
                    'handle' => $taxonomy->handle(),
                    'title' => $taxonomy->title(),
                ];
                })->toArray(),
        ];

        $crumbs = Breadcrumbs::make([
            ['text' => 'Statamic Site Kit', 'url' => url(config('statamic.cp.route') . '/statamic-site-kit/seo')],
            ['text' => 'Content Defaults', 'url' => url(config('statamic.cp.route') . '/statamic-site-kit/seo/defaults')],
        ]);

        return view('statamic-site-kit::cp.seo.defaults.index', [
            'content_types' => $content_types,
            'crumbs' => $crumbs,
            'title' => 'Content Defaults | Statamic Site Kit',
        ]);
    }

    /**
     * Return the view for editing individual content type's content type
     *
     * @param Illuminate\Http\Request $request
     * @param string $content_type
     */
    public function edit(Request $request, string $content_type)
    {
        $this->authorize('view ssk seo defaults');

        $data = $this->getData($content_type);

        $blueprint = $this->getBlueprint();
        $fields = $blueprint->fields()->addValues($data)->preProcess();

        $repo = $this->getRepositoryFromHandle($content_type);

        $crumbs = Breadcrumbs::make([
            ['text' => __('statamic-site-kit::main.name'), 'url' => cp_route('statamic-site-kit.home')],
            ['text' => __('statamic-site-kit::main.seo'), 'url' => cp_route('statamic-site-kit.seo.home')],
            ['text' => __('statamic-site-kit::defaults.index'), 'url' => cp_route('statamic-site-kit.seo.defaults.index')],
            ['text' => $repo->title(), 'url' => cp_route('statamic-site-kit.seo.defaults.edit', ['default' => $content_type])],
        ]);

        return view('statamic-site-kit::cp.seo.defaults.edit', [
            'blueprint' => $blueprint->toPublishArray(),
            'crumbs' => $crumbs,
            'meta' => $fields->meta(),
            'title' => "{$repo->title()} Defaults | Statamic Site Kit",
            'repo' => $repo,
            'content_type' => $content_type,
            'values' => $fields->values(),
        ]);
    }

    /**
     * Save the defaults data for this content type
     *
     * @param Illuminate\Http\Request $request
     * @param string $content_type
     */
    public function update(Request $request, string $content_type)
    {
        $this->authorize('update ssk seo defaults');

        $blueprint = $this->getBlueprint();

        $fields = $blueprint->fields()->addValues($request->all());
        $fields->validate();

        $this->putData($content_type, $fields->process()->values()->toArray());

        $content_type_parts = explode('_', $content_type, 2);
        StatamicSiteKitContentDefaultsSaved::dispatch(new ContentDefaults($content_type_parts[0], $content_type_parts[1], Site::selected()));
    }

    public function getBlueprint()
    {
        return DefaultsBlueprint::requestBlueprint();
    }

    /**
     * Get the data from the relevant defaults file
     *
     * @param string $content_type
     *
     * @return array
     */
    public function getData(string $content_type)
    {
        return StatamicSiteKitStorage::getYaml("defaults/{$content_type}", Site::selected());
    }

    /**
     * Set the data for a single content type
     *
     * @param string $content_type
     * @param array $data
     */
    public function putData(string $content_type, array $data)
    {
        StatamicSiteKitStorage::putYaml("defaults/{$content_type}", Site::selected(), $data);
    }

    /**
     * Return the content repository from our generated handle
     *
     * @param string $handle
     */
    private function getRepositoryFromHandle(string $handle)
    {
        $parts = explode('_', $handle);
        $type = array_shift($parts);
        $content_handle = implode('_', $parts);

        if ($type === 'collections') {
            return Collection::findByHandle($content_handle);
        } elseif ($type === 'taxonomies') {
            return Taxonomy::findByHandle($content_handle);
        }
    }
}
