<?php

namespace Michaelravedoni\StatamicSiteKit\Tags;

use Statamic\Facades\Entry;
use Statamic\Facades\Site;
use Statamic\Tags\Tags;
use Statamic\View\View;
use Michaelravedoni\StatamicSiteKit\Schema\SchemaGraph;
use Michaelravedoni\StatamicSiteKit\Facades\PageDataParser;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Seo\DefaultsBlueprint;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Seo\GeneralBlueprint;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Seo\MarketingBlueprint;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Seo\SocialBlueprint;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Seo\OnPageSeoBlueprint;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Settings\InformationsBlueprint;
use Michaelravedoni\StatamicSiteKit\Blueprints\CP\Settings\ManifestsBlueprint;

class StatamicSiteKitTags extends Tags
{
    protected static $handle = 'ssk';

    /**
     * Return all data
     */
    public function data()
    {
        $informations = PageDataParser::getValues('informations', new InformationsBlueprint);
        $manifests = PageDataParser::getValues('manifests', new ManifestsBlueprint);
        $marketing = PageDataParser::getValues('marketing', new MarketingBlueprint);
        $general = PageDataParser::getValues('general', new GeneralBlueprint);
        $data = collect()
            ->merge($manifests)
            ->merge($informations)
            ->merge($marketing)
            ->merge($general);
        return $data;
    }

    /**
     * Return manifests data
     */
    public function manifests()
    {
        $data = PageDataParser::getValues('manifests', new ManifestsBlueprint);
        return $data;
    }

    /**
     * Return the <head /> tag content required for on-page SEO
     *
     * @return string
     */
    public function head()
    {
        $data = PageDataParser::getData(collect($this->context));

        $view = View::make('statamic-site-kit::tags.head', $data->all());

        if ($this->params->get('debug')) {
            return $view;
        }

        return preg_replace(
            [
                "/<!--(.|\s)*?-->/",
                "/(^[\r\n]*|[\r\n]+)[\s\t]*[\r\n]+/",
            ],
            [
                '',
                "\n",
            ],
            $view
        );
    }

    /**
     * Return the body content
     */
    public function body()
    {
        $data = PageDataParser::getData(collect($this->context));
        return View::make('statamic-site-kit::tags.body', $data->all());
    }

    /**
     * Return the footer content
     */
    public function footer()
    {
        $data = PageDataParser::getData(collect($this->context));
        return View::make('statamic-site-kit::tags.footer', $data->all());
    }

    /**
     * Return the data for hreflang tags
     */
    public function hreflang()
    {
        $ctx = collect($this->context);

        $id = $ctx->get('id');

        if ($id instanceof \Statamic\Fields\Value) {
            $id = $id->value();
        }

        if (empty($id)) {
            return null;
        }

        $data = Entry::find($id);

        if (!$data) {
            return null;
        }

        $sites_by_handle = Site::all()->reduce(function($sites, $site) {
            $sites[$site->handle()] = $site;
            return $sites;
        }, []);

        $alternates = $data->sites()->map(function ($handle) use ($data, $sites_by_handle) {
            $localized_data = $data->in($handle);

            if(!empty($localized_data) && $localized_data->published()) {
                $site = $sites_by_handle[$handle];
                return [
                    'url' => $localized_data->absoluteUrl(),
                    'locale' => $site->locale()
                ];
            }

            return null;
        }, [])->filter();

        if (!empty($alternates)) {
            return view('statamic-site-kit::tags.hreflang', [
                'hreflang_tags' => $alternates,
            ]);
        }
    }

    /**
     * Return the schema graph object
     *
     * @return string
     */
    public function graph()
    {
        $ctx = collect($this->context);
        $graph = new SchemaGraph($ctx);
        return $graph->build();
    }

    /**
     * Return the list of social icons created in the 'Social' menu
     *
     * @return string
     */
    public function socials()
    {
        $socials = PageDataParser::getValues('informations', new InformationsBlueprint)->get('social_links');
        if ($socials->raw()) {
            return $this->parseLoop($socials->raw());
        }
        return $this->parseLoop([]);
    }

    /**
     * Return the list of social icons created in the 'Social' menu
     *
     * @return string
     */
    public function socialsSeo()
    {
        $data = PageDataParser::getData(collect($this->context));
        $socials = $data->get('ssk_social')->get('seo_social_links');
        if ($socials->raw()) {
            return $this->parseLoop($socials->raw());
        }
        return $this->parseLoop([]);
    }

    /**
     * Return the robots tag content
     * (done here to prevent a bunch of ifs and butts in the template file)
     *
     * @return string
     */
    public function robotsTag()
    {
        $ctx = collect($this->context);
        $attrs = [];

        $global_no_index = $ctx->get('ssk_general')['no_index_site'];

        if ($ctx->get('no_index_page') || $global_no_index->raw()) {
            array_push($attrs, 'noindex');
        }

        if ($ctx->get('no_follow_links')) {
            array_push($attrs, 'nofollow');
        }

        if (!empty($attrs)) {
            $attrs_string = implode(', ', $attrs);
            return "<meta name=\"robots\" content=\"{$attrs_string}\">";
        }

        return false;
    }

    /**
     * Return a generated canonical URL - this should contain pagination vars
     * if any are set
     *
     * @return string
     */
    public function generatedCanonical()
    {
        $data = collect($this->context);
        $vars = $data->get('get');
        $current_url = $data->get('permalink');
        if ($vars && $page = collect($vars)->get('page')) {
            $current_url .= '?page=' . urlencode($page);
        }
        return $current_url;
    }
}
